/*
 * Copyright (C) 2014 Canonical Ltd.
 *
 * This file is part of unity-chromium-extension
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

extern "C" {
#include <unity-webapps-permissions.h>
#include <unity-webapps-application-repository.h>
}

#include "webapps-handler.h"

#include <QDebug>
#include <QRegularExpression>


using namespace UnityWebapps;

namespace UnityWebapps {

class WebappsHandlerPrivate
{
    Q_DECLARE_PUBLIC(WebappsHandler)

public:
    inline WebappsHandlerPrivate(WebappsHandler *service);
    inline ~WebappsHandlerPrivate();

    UnityWebappsApplicationRepository *getRepository();

private:
    static
    void installationCallback(UnityWebappsApplicationRepository *repository,
                              const gchar *name,
                              UnityWebappsApplicationStatus status,
                              gpointer user_data);

    mutable WebappsHandler *q_ptr;
    UnityWebappsApplicationRepository *applicationRepository;
};

} // namespace


WebappsHandlerPrivate::WebappsHandlerPrivate(WebappsHandler *service):
    q_ptr(service),
    applicationRepository(0)
{
}

WebappsHandlerPrivate::~WebappsHandlerPrivate()
{
    if (applicationRepository != 0) {
        g_object_unref(G_OBJECT(applicationRepository));
    }
}

UnityWebappsApplicationRepository *WebappsHandlerPrivate::getRepository()
{
    if (applicationRepository == 0) {
        applicationRepository = unity_webapps_application_repository_new_default();
        unity_webapps_application_repository_prepare(applicationRepository);
    }
    return applicationRepository;
}

void WebappsHandlerPrivate::installationCallback(
    UnityWebappsApplicationRepository *repository,
    const gchar *name,
    UnityWebappsApplicationStatus status,
    gpointer user_data
)
{
    Q_UNUSED(repository);
    Q_UNUSED(name);
    Q_UNUSED(status);
    Q_UNUSED(user_data);
}


WebappsHandler::WebappsHandler(QObject *parent):
    QObject(parent),
    d_ptr(new WebappsHandlerPrivate(this))
{
}

WebappsHandler::~WebappsHandler()
{
    delete d_ptr;
}

QVariantMap WebappsHandler::url_loaded(const QVariantMap &message)
{
    Q_D(WebappsHandler);

    QVariantMap reply;

    if (!message.contains("url")) {
        reply.insert("error", QStringLiteral("malformed request"));
        return reply;
    }

    QString url = message.value("url").toString();

    GList *apps = unity_webapps_application_repository_resolve_url(
                    d->getRepository(),
                    url.toLocal8Bit().constData()
                  );

    if (apps == 0) {
        reply.insert("available", false);
        return reply;
    }

    // As per original extension only look at the first entry
    const gchar *packageName = (const gchar*)apps->data;
    UnityWebappsApplicationStatus status = unity_webapps_application_repository_get_resolved_application_status(
                d->getRepository(),
                packageName);

    if (status == UNITY_WEBAPPS_APPLICATION_STATUS_UNRESOLVED) {
        reply.insert("available", false);
        g_list_free_full(apps, g_free);
        return reply;
    }

    const gchar *appName = unity_webapps_application_repository_get_resolved_application_name(
                        d->getRepository(),
                        packageName);

    const gchar *appDomain = unity_webapps_application_repository_get_resolved_application_domain(
                        d->getRepository(),
                        packageName);

    if (unity_webapps_permissions_get_domain_dontask(appDomain) ||
        unity_webapps_permissions_get_domain_allowed(appDomain)) {
        reply.insert("available", false);
        g_list_free_full(apps, g_free);
        return reply;
    }

    reply.insert("available", true);
    reply.insert("appName", QString::fromUtf8(appName));
    reply.insert("appDomain", QString::fromUtf8(appDomain));

    g_list_free_full(apps, g_free);

    return reply;
}

QVariantMap WebappsHandler::dont_ask(const QVariantMap &message)
{
    Q_D(WebappsHandler);

    QVariantMap reply;

    if (!message.contains("url")) {
        reply.insert("error", QLatin1String("malformed request"));
        return reply;
    }

    QString url = message.value("url").toString();

    GList *apps = unity_webapps_application_repository_resolve_url(
                    d->getRepository(),
                    url.toLocal8Bit().constData()
                  );

    if (apps == 0) {
        reply.insert("available", false);
        reply.insert("dont_ask", false);
        return reply;
    }

    const gchar *packageName = (const gchar*)apps->data;

    const gchar *appDomain = unity_webapps_application_repository_get_resolved_application_domain(
                        d->getRepository(),
                        packageName);

    unity_webapps_permissions_dontask_domain(appDomain);

    reply.insert("dont_ask", true);

    g_list_free_full(apps, g_free);

    return reply;
}

QVariantMap WebappsHandler::install(const QVariantMap &message)
{
    Q_D(WebappsHandler);

    QVariantMap reply;

    if (!message.contains("url")) {
        reply.insert("error", QLatin1String("malformed request"));
        return reply;
    }

    QString url = message.value("url").toString();

    GList *apps = unity_webapps_application_repository_resolve_url(
                    d->getRepository(),
                    url.toLocal8Bit().constData()
                  );

    if (apps == 0) {
        reply.insert("available", false);
        reply.insert("installed", false);
        return reply;
    }

    const gchar *packageName = (const gchar*)apps->data;

    UnityWebappsApplicationStatus status = unity_webapps_application_repository_get_resolved_application_status(
                d->getRepository(),
                packageName);

    if (status == UNITY_WEBAPPS_APPLICATION_STATUS_UNRESOLVED) {
        reply.insert("available", false);
        reply.insert("installed", false);
        g_list_free_full(apps, g_free);
        return reply;
    }

    const gchar *appDomain = unity_webapps_application_repository_get_resolved_application_domain(
                        d->getRepository(),
                        packageName);
    const gchar *appName = unity_webapps_application_repository_get_resolved_application_name(
                        d->getRepository(),
                        packageName);

    switch (status) {
    case UNITY_WEBAPPS_APPLICATION_STATUS_AVAILABLE:
        if (unity_webapps_permissions_get_domain_dontask(appDomain)) {
            reply.insert("available", true);
            reply.insert("installed", false);
            g_list_free_full(apps, g_free);
            return reply;
        }
        unity_webapps_permissions_allow_domain(appDomain);
        unity_webapps_application_repository_install_application(
            d->getRepository(),
            packageName,
            WebappsHandlerPrivate::installationCallback,
            0);
        break;

    case UNITY_WEBAPPS_APPLICATION_STATUS_INSTALLED:
        if (unity_webapps_permissions_get_domain_dontask(appDomain) ||
            unity_webapps_permissions_get_domain_allowed(appDomain)) {
            reply.insert("available", true);
            reply.insert("installed", true);
            g_list_free_full(apps, g_free);
            return reply;
        }
        unity_webapps_application_repository_add_desktop_to_launcher(
                createApplicationDesktopName(appName, appDomain).toLocal8Bit()
            );
        unity_webapps_permissions_allow_domain(appDomain);
        break;
    case UNITY_WEBAPPS_APPLICATION_STATUS_UNRESOLVED:
        // handled above
        break;
    }

    reply.insert("installed", true);

    g_list_free_full(apps, g_free);

    return reply;
}

QVariantMap WebappsHandler::ping(const QVariantMap &message)
{
    Q_UNUSED(message);

    QVariantMap reply;

    reply.insert("pong", QLatin1String("alive"));

    return reply;
}

/*
    \fn createApplicationDesktopName(QString, QString)

    This function creates a application:// URI suitable for passing to
    unity_webapps_application_repository_add_desktop_to_launcher().

    To add the webapp's icon to the launcher, a dbus message is sent to the
    launcher, referencing the webapps canonical .desktop file name. This
    function creates the canonical name and returns it in the URI form.
*/

QString WebappsHandler::createApplicationDesktopName(
    const QString &appName,
    const QString &appDomain
) const
{
    QString basename;
    QRegularExpression re(QStringLiteral("[^[:alnum:]]"),
                            QRegularExpression::UseUnicodePropertiesOption);

    basename.append(appName.trimmed().replace(re, QStringLiteral("")));
    basename.append(appDomain.trimmed().replace(re, QStringLiteral("")));

    return QStringLiteral("application://%1.desktop").arg(basename);
}

