# /*##########################################################################
# Copyright (C) 2018 European Synchrotron Radiation Facility
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# ############################################################################*/
"""
This module provides implementations based on marching squares algorithms.

The main implementation is done by :class:`MarchingSquaresMergeImpl`. It was
designed to speed up the computation of iso surface using Cython and OpenMP.
It also provides features like support of mask, and cache of min/max per tiles
which is very efficient to find many iso contours from image gradient.

Utilitary functions are provided as facade for simple use.
:meth:`find_contours` to find iso contours from an image and using the same
main signature as `find_contours` from `skimage`, but supporting mask.
And :meth:`find_pixels` which returns a set of pixel coords containing the
points of the iso contours.
"""

__authors__ = ["V. Valls"]
__license__ = "MIT"
__date__ = "02/07/2018"


from ._mergeimpl import MarchingSquaresMergeImpl


def _factory(engine, image, mask):
    """Factory to create the marching square implementation from the engine
    name"""
    if engine == "merge":
        return MarchingSquaresMergeImpl(image, mask)
    elif engine == "skimage":
        from _skimage import MarchingSquaresSciKitImage

        return MarchingSquaresSciKitImage(image, mask)
    else:
        raise ValueError(
            "Engine '%s' is not supported ('merge' or 'skimage' expected)."
        )


def find_pixels(image, level, mask=None):
    """
    Find the pixels following the iso contours at the given `level`.

    These pixels are localized by the bound of the segment generated by the
    iso contour algorithm.

    The result is returned as a numpy array storing a list of coordinates y/x.

    .. code-block:: python

        # Example using a mask
        shape = 100, 100
        image = numpy.random.random(shape)
        mask = numpy.random.random(shape) < 0.01
        pixels = silx.image.marchingsquares.find_pixels(image, 0.5, mask=mask)

    :param numpy.ndarray image: Image to process
    :param float level: Level of the requested iso contours.
    :param numpy.ndarray mask: An optional mask (a non-zero value invalidate
        the pixels of the image)
    :returns: An array of coordinates in y/x
    :rtype: numpy.ndarray
    """
    assert image is not None
    if mask is not None:
        assert image.shape == mask.shape
    engine = "merge"
    impl = _factory(engine, image, mask)
    return impl.find_pixels(level)


def find_contours(image, level, mask=None):
    """
    Find the iso contours at the given `level`.

    The result is returned as a list of polygons.

    .. code-block:: python

        # Example using a mask
        shape = 100, 100
        image = numpy.random.random(shape)
        mask = numpy.random.random(shape) < 0.01
        polygons = silx.image.marchingsquares.find_contours(image, 0.5, mask=mask)

    :param numpy.ndarray image: Image to process
    :param float level: Level of the requested iso contours.
    :param numpy.ndarray mask: An optional mask (a non-zero value invalidate
        the pixels of the image)
    :returns: A list of array containing y-x coordinates of points
    :rtype: List[numpy.ndarray]
    """
    assert image is not None
    if mask is not None:
        assert image.shape == mask.shape
    engine = "merge"
    impl = _factory(engine, image, mask)
    return impl.find_contours(level)
