///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __ABSTRACT_FILE_COLUMN_PARSER_H
#define __ABSTRACT_FILE_COLUMN_PARSER_H

#include <core/Core.h>
#include "AtomsFileParser.h"
#include "ColumnChannelMapping.h"

namespace AtomViz {

/**
 * \brief Abstract base class for atom file format parsers that can
 *        handle a variable number of data columns with a customizable
 *        mapping to data channels.
 *
 * This abstract base class is meant as basis for specialized file format
 * parsers that read atomistic data from a data file with tabular structure 
 * (like the simple XYZ format).
 * 
 * This base class stores the mapping between the data columns in the input file 
 * and the data channels (DataChannel) in the loaded AtomsObject.
 * 
 * \sa DataChannel, AtomsObject, ColumnChannelMapping, DataRecordParserHelper
 * \author Alexander Stukowski 
 */ 
class ATOMVIZ_DLLEXPORT AbstractFileColumnParser : public AtomsFileParser
{
protected:

	/// \brief Constructs a new instance of this class. 
	/// \param isLoading Specifies whether the object's data fields will be initialized from the 
	///                  data stored in a scene file after the instance has been created.
	AbstractFileColumnParser(bool isLoading = false) : AtomsFileParser(isLoading) {}

public:

	/// \brief Returns the mapping between data columns in the input file and
	///        the DataChannel objects in the final AtomsObject. 
	const ColumnChannelMapping& columnMapping() const { return _columnMapping; }
	
	/// \brief Sets the mapping between data columns in the input file and
	///        the DataChannel objects in the final AtomsObject.
	void setColumnMapping(const ColumnChannelMapping& mapping);

	// From AtomsFileParser:

	/// \brief Sets the name of the input file for this parser.
	virtual bool setInputFile(const QString& filename);
	
protected:

	/// \brief Saves the class' contents to the given stream.
	/// \sa RefTarget::saveToStream() 
	virtual void saveToStream(ObjectSaveStream& stream);
	
	/// \brief Loads the class' contents from the given stream.
	/// \sa RefTarget::loadFromStream() 
	virtual void loadFromStream(ObjectLoadStream& stream);		

	/// \brief Creates a copy of this object.
	/// \sa RefTarget::Clone() 
	virtual RefTarget::SmartPtr clone(bool deepCopy, CloneHelper& cloneHelper);	
	
	/// \brief Parses the header of the given file and returns the number of data columns contained in the file.
	/// \param[in] filename The filepath of the atoms file that should be inspected.
	/// \param[out] numberOfColumns The method must set this reference variable to the number of columns found in the atoms file.
	/// \param[out] columnNames The method should fill this list with the names of the columns. If the atoms file does not contains
	///                         this information then the list should be left empty.
	/// \return \a false when the operation has been canceled by the user; \a true on success.
	/// \throws Exception on parsing error.	
	virtual bool inspectFileHeader(const QString& filename, int& numberOfColumns, QStringList& columnNames) = 0;
	
private:

	/// Stores the mapping between data columns in the input file and
	/// the DataChannel objects in the final AtomsObject. 
	ColumnChannelMapping _columnMapping;
	
private:
	Q_OBJECT
	DECLARE_ABSTRACT_PLUGIN_CLASS(AbstractFileColumnParser)	
};


};	// End of namespace AtomViz

#endif // __ABSTRACT_FILE_COLUMN_PARSER_H
