/*
  SPDX-FileCopyrightText: 2025 Laurent Montel <montel@kde.org>

  SPDX-License-Identifier: GPL-2.0-or-later
*/

#include "textautogeneratedelegateutils.h"

#include "textautogeneratelistviewtextselection.h"

#include <QClipboard>
#include <QTextDocument>
#include <QTextFormat>
#include <QTextStream>
#include <qabstracttextdocumentlayout.h>

using namespace Qt::Literals::StringLiterals;
using namespace TextAutoGenerateText;

int TextAutoGenerateDelegateUtils::spacingText()
{
    return 10;
}

int TextAutoGenerateDelegateUtils::roundRectValue()
{
    return 20;
}

int TextAutoGenerateDelegateUtils::marginText()
{
    return 20;
}

int TextAutoGenerateDelegateUtils::rightIndent()
{
    return 30;
}

int TextAutoGenerateDelegateUtils::leftLLMIndent()
{
    return rightIndent();
}

int TextAutoGenerateDelegateUtils::leftUserIndent()
{
    return 80;
}

int TextAutoGenerateDelegateUtils::iconSpacing()
{
    return 10;
}

QSize TextAutoGenerateDelegateUtils::progressIndicatorSize()
{
    return QSize(60, 20);
}

bool TextAutoGenerateDelegateUtils::generateToolTip(const QTextDocument *doc, const QPoint &pos, QString &formattedTooltip)
{
    const auto format = doc->documentLayout()->formatAt(pos);
    const auto tooltip = format.property(QTextFormat::TextToolTip).toString();
    const auto href = format.property(QTextFormat::AnchorHref).toString();
    if (href.isEmpty() && tooltip.isEmpty()) {
        return false;
    }
    generateToolTip(tooltip, href, formattedTooltip);
    return true;
}

void TextAutoGenerateDelegateUtils::generateToolTip(const QString &toolTip, const QString &href, QString &formattedTooltip)
{
    QTextStream stream(&formattedTooltip);
    auto addLine = [&](const QString &line) {
        if (!line.isEmpty()) {
            stream << "<p>"_L1 << line << "</p>"_L1;
        }
    };

    stream << "<qt>"_L1;
    addLine(toolTip);
    addLine(href);
    stream << "</qt>"_L1;
}

QList<QAbstractTextDocumentLayout::Selection> TextAutoGenerateDelegateUtils::selection(TextAutoGenerateListViewTextSelection *selection,
                                                                                       QTextDocument *doc,
                                                                                       const QModelIndex &index,
                                                                                       const QStyleOptionViewItem &option)
{
    QList<QAbstractTextDocumentLayout::Selection> selections;
    const QTextCursor selectionTextCursor = selection->selectionForIndex(index, doc);
    if (!selectionTextCursor.isNull()) {
        QTextCharFormat selectionFormat;
        selectionFormat.setBackground(option.palette.brush(QPalette::Highlight));
        selectionFormat.setForeground(option.palette.brush(QPalette::HighlightedText));
        selections.append({selectionTextCursor, selectionFormat});
    }
    return selections;
}

void TextAutoGenerateDelegateUtils::setClipboardSelection(TextAutoGenerateListViewTextSelection *selection)
{
    QClipboard *clipboard = QGuiApplication::clipboard();
    if (selection->hasSelection() && clipboard->supportsSelection()) {
        const QString text = selection->selectedText(TextAutoGenerateListViewTextSelection::Format::Text);
        clipboard->setText(text, QClipboard::Selection);
    }
}
