/* $Id$ 
 *
 * SetParam: Set a parameter in the transfer region.
 *
 * Copyright (C) 2008-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */
#ifndef __SET_PARAM_HPP_INCLUDED
#define __SET_PARAM_HPP_INCLUDED

#include "intermediate/opcodes/OpCode.hpp"
#include "intermediate/operands/Reference.hpp"

namespace intermediate {

//! Copy source to a slot in the transfer region of a container.
/** This opcode will copy source to the transfer slot dst from container
 *  container into the previously allocated region.
 *
 *  Read Operands: src, cnt, dst
 *  Write operands: no publicly visible operands.
 *
 *  Operation:
 *      copy src to the slot defined by cnt/dst.
 *
 *  Errors:
 *      SetParam must be evaluated after a corresponding BeginTransfer
 *      statement in the same text segment before a corresponding
 *      EndTransfer statement.
 *
 *      The Reference to the container must match the Reference to 
 *      BeginTransfer.
 *
 *      Hint: An implementation could allocate space on the stack for
 *      BeginTransfer (where the size would match the size of the transfer
 *      statement), SetParam could calculate the relative stack offset
 *      by cnt/dest to the stack top and EndTransfer could deallocate the
 *      space on the stack again. So SetParam w.o. BeginTransfer would
 *      basically corrupt the current stack frame.
 */
class SetParam : public OpCode {
public:
	//! c'tor
	/** @param source source operand
	 *  @param cnt Reference to the container
	 *  @param dest Reference to item in transfer segment of container.
	 */
	SetParam(
		Operand *source, 
		Reference *cnt, 
		Reference *dest
		) :	src(source),
			container(cnt),
			dst(dest) {}

	//! Accept a Visitor.
 	/** All intermediate code nodes need to implement this method.
         *
         *  @param v the Visitor that can visit this node.
         */
	virtual void accept(Visitor& v) {
		v.visit(*this);
	}

	//! source operand
	Operand *src;
	//! referenced container
	Reference *container;
	//! item in transfer segment of referenced container
	Reference *dst;

protected:
	virtual ~SetParam() {
		util::MiscUtil::terminate(this->src);
		util::MiscUtil::terminate(this->container);
		util::MiscUtil::terminate(this->dst);
	}
};

}; /* namespace intermediate */

#endif /* __SET_PARAM_HPP_INCLUDED */
