#include <assert.h>
#include <fcntl.h>
#include <stdlib.h>
#include <sys/types.h>

#include "client.h"
#include "log.h"
#include "server.h"
#include "serverpool.h"
#include "state.pb.h"
#include "udpstream.h"
#include "util.h"

struct sockaddr_in6;

using namespace std;

ServerPool::ServerPool(int size)
	: servers(new Server[size]),
	  num_servers(size)
{
}

CubemapStateProto ServerPool::serialize()
{
	CubemapStateProto state;

	unordered_map<const string *, size_t> short_response_pool;

	for (int i = 0; i < num_servers; ++i) {
                CubemapStateProto local_state = servers[i].serialize(&short_response_pool);

		// The stream state should be identical between the servers, so we only store it once,
		// save for the fds, which we keep around to distribute to the servers after re-exec.
		if (i == 0) {
			state.mutable_streams()->MergeFrom(local_state.streams());
		} else {
			assert(state.streams_size() == local_state.streams_size());
			for (int j = 0; j < local_state.streams_size(); ++j) {
				assert(local_state.streams(j).data_fds_size() == 1);
				state.mutable_streams(j)->add_data_fds(local_state.streams(j).data_fds(0));
			}
		}
		for (const ClientProto &client : local_state.clients()) {
			state.add_clients()->MergeFrom(client);
		}
		for (const HLSZombieProto &hls_zombie : local_state.hls_zombies()) {
			state.add_hls_zombies()->MergeFrom(hls_zombie);
		}
        }

	for (size_t i = 0; i < short_response_pool.size(); ++i) {
		state.mutable_short_response_pool()->Add();
	}
	for (const auto &string_and_index : short_response_pool) {
		state.mutable_short_response_pool(string_and_index.second)->set_header_or_short_response(*string_and_index.first);
	}

	return state;
}

void ServerPool::add_client(int sock, Acceptor *acceptor)
{
	servers[clients_added++ % num_servers].add_client_deferred(sock, acceptor);
}

void ServerPool::add_client_from_serialized(const ClientProto &client, const std::vector<std::shared_ptr<const std::string>> &short_responses)
{
	servers[clients_added++ % num_servers].add_client_from_serialized(client, short_responses);
}

// It's fine to abuse clients_added here, since it's only ever used for round-robin purposes.
void ServerPool::add_hls_zombie_from_serialized(const HLSZombieProto &hls_zombie)
{
	servers[clients_added++ % num_servers].add_hls_zombie_from_serialized(hls_zombie);
}

int ServerPool::lookup_stream_by_url(const string &url) const
{
	assert(servers != nullptr);
	return servers[0].lookup_stream_by_url(url);
}

int ServerPool::add_stream(const string &url,
                           const string &hls_url,
                           size_t backlog_size,
                           size_t prebuffering_bytes,
                           Stream::Encoding encoding,
                           Stream::Encoding src_encoding,
                           unsigned hls_frag_duration,
                           size_t hls_backlog_margin,
                           const string &allow_origin)
{
	// Adding more HTTP streams after UDP streams would cause the UDP stream
	// indices to move around, which is obviously not good.
	assert(udp_streams.empty());

	for (int i = 0; i < num_servers; ++i) {
		int stream_index = servers[i].add_stream(url, hls_url, backlog_size, prebuffering_bytes, encoding, src_encoding, hls_frag_duration, hls_backlog_margin, allow_origin);
		assert(stream_index == num_http_streams);
	}
	return num_http_streams++;
}

int ServerPool::add_stream_from_serialized(const StreamProto &stream, const vector<int> &data_fds)
{
	// Adding more HTTP streams after UDP streams would cause the UDP stream
	// indices to move around, which is obviously not good.
	assert(udp_streams.empty());

	assert(!data_fds.empty());
	string contents;
	for (int i = 0; i < num_servers; ++i) {
		int data_fd;
		if (i < int(data_fds.size())) {
			// Reuse one of the existing file descriptors.
			data_fd = data_fds[i];
		} else {
			// Clone the first one.
			if (contents.empty()) {
				if (!read_tempfile(data_fds[0], &contents)) {
					exit(1);
				}
			}
			data_fd = make_tempfile(contents);
		}

		int stream_index = servers[i].add_stream_from_serialized(stream, data_fd);
		assert(stream_index == num_http_streams);
	}

	// Close and delete any leftovers, if the number of servers was reduced.
	for (size_t i = num_servers; i < data_fds.size(); ++i) {
		safe_close(data_fds[i]);  // Implicitly deletes the file.
	}

	return num_http_streams++;
}
	
int ServerPool::add_udpstream(const sockaddr_in6 &dst, int pacing_rate, int ttl, int multicast_iface_index)
{
	udp_streams.emplace_back(new UDPStream(dst, pacing_rate, ttl, multicast_iface_index));
	return num_http_streams + udp_streams.size() - 1;
}

void ServerPool::set_header(int stream_index, const string &http_header, const string &stream_header)
{
	assert(stream_index >= 0 && stream_index < ssize_t(num_http_streams + udp_streams.size()));

	if (stream_index >= num_http_streams) {
		// UDP stream. TODO: Log which stream this is.
		if (!stream_header.empty()) {
			log(WARNING, "Trying to send stream format with headers to a UDP destination. This is unlikely to work well.");
		}

		// Ignore the HTTP header.
		return;
	}

	// HTTP stream.
	for (int i = 0; i < num_servers; ++i) {
		servers[i].set_header(stream_index, http_header, stream_header);
	}
}

void ServerPool::add_data(int stream_index, const char *data, size_t bytes, uint16_t metacube_flags, const RationalPTS &pts)
{
	assert(stream_index >= 0 && stream_index < ssize_t(num_http_streams + udp_streams.size()));

	if (stream_index >= num_http_streams) {
		// UDP stream.
		udp_streams[stream_index - num_http_streams]->send(data, bytes);
		return;
	}

	// HTTP stream.
	for (int i = 0; i < num_servers; ++i) {
		servers[i].add_data_deferred(stream_index, data, bytes, metacube_flags, pts);
	}
}

void ServerPool::set_unavailable(int stream_index)
{
	assert(stream_index >= 0 && stream_index < ssize_t(num_http_streams + udp_streams.size()));
	if (stream_index < ssize_t(num_http_streams)) {
		for (int i = 0; i < num_servers; ++i) {
			servers[i].set_unavailable(stream_index);
		}
	}
}

void ServerPool::add_gen204(const std::string &url, const std::string &allow_origin)
{
	for (int i = 0; i < num_servers; ++i) {
		servers[i].add_gen204(url, allow_origin);
	}
}

void ServerPool::create_tls_context_for_acceptor(const Acceptor *acceptor)
{
	for (int i = 0; i < num_servers; ++i) {
		servers[i].create_tls_context_for_acceptor(acceptor);
	}
}

void ServerPool::run()
{
	for (int i = 0; i < num_servers; ++i) {
		servers[i].run();
	}
}
	
void ServerPool::stop()
{
	for (int i = 0; i < num_servers; ++i) {
		servers[i].stop();
	}
}
	
vector<ClientStats> ServerPool::get_client_stats() const
{
	vector<ClientStats> ret;
	for (int i = 0; i < num_servers; ++i) {
		vector<ClientStats> stats = servers[i].get_client_stats();
		ret.insert(ret.end(), stats.begin(), stats.end());
	}
	return ret;
}

vector<HLSZombie> ServerPool::get_hls_zombies() const
{
	vector<HLSZombie> ret;
	for (int i = 0; i < num_servers; ++i) {
		vector<HLSZombie> stats = servers[i].get_hls_zombies();
		ret.insert(ret.end(), stats.begin(), stats.end());
	}
	return ret;
}

void ServerPool::set_pacing_rate(int stream_index, uint32_t pacing_rate)
{
	for (int i = 0; i < num_servers; ++i) {
		servers[i].set_pacing_rate(stream_index, pacing_rate);
	}	
}

void ServerPool::set_backlog_size(int stream_index, size_t new_size)
{
	for (int i = 0; i < num_servers; ++i) {
		servers[i].set_backlog_size(stream_index, new_size);
	}	
}

void ServerPool::set_prebuffering_bytes(int stream_index, size_t new_amount)
{
	for (int i = 0; i < num_servers; ++i) {
		servers[i].set_prebuffering_bytes(stream_index, new_amount);
	}
}

void ServerPool::set_encoding(int stream_index, Stream::Encoding encoding)
{
	for (int i = 0; i < num_servers; ++i) {
		servers[i].set_encoding(stream_index, encoding);
	}	
}

void ServerPool::set_src_encoding(int stream_index, Stream::Encoding encoding)
{
	for (int i = 0; i < num_servers; ++i) {
		servers[i].set_src_encoding(stream_index, encoding);
	}
}

void ServerPool::set_hls_frag_duration(int stream_index, unsigned hls_frag_duration)
{
	for (int i = 0; i < num_servers; ++i) {
		servers[i].set_hls_frag_duration(stream_index, hls_frag_duration);
	}
}

void ServerPool::set_hls_backlog_margin(int stream_index, size_t hls_backlog_margin)
{
	for (int i = 0; i < num_servers; ++i) {
		servers[i].set_hls_backlog_margin(stream_index, hls_backlog_margin);
	}
}

void ServerPool::set_allow_origin(int stream_index, const std::string &allow_origin)
{
	for (int i = 0; i < num_servers; ++i) {
		servers[i].set_allow_origin(stream_index, allow_origin);
	}
}

void ServerPool::register_hls_url(int stream_index, const string &hls_url)
{
	for (int i = 0; i < num_servers; ++i) {
		servers[i].register_hls_url(stream_index, hls_url);
	}
}
