#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <vector>

#include <unistd.h>
#include <openssl/ocsp.h>
#include <openssl/ossl_typ.h>
#include <openssl/rand.h>
#include <openssl/asn1.h>

#include "opensslutil.h"
#include "opensslgenericutil.h"
#include "canlxx.h"

namespace AuthN {
namespace OpenSSL {

  #define MAX_RESPONSE_STR 5120
  #define MAX_REQUEST_STR 2048

  static Status get_cert_id(Context* ctx, X509* cert, X509** issuer, OCSP_CERTID** cert_id) {
    X509_STORE_CTX store_ctx;
    X509_STORE* cert_store = NULL;
    if(*issuer) { X509_free(*issuer); *issuer = NULL; }
   
    if(!cert) return Status(-1, "X509 object is empty");
    if(!ctx) return Status(-1, "Context object is empty");
   
    std::string cadir = ctx->GetCAPath();
    cert_store = setup_verify(*ctx, "", cadir);

    // create OCSP_CERTID
    if(!cert_store) return Status(-1, "Failed to create X509_STORE");
    if (!X509_STORE_CTX_init(&store_ctx, cert_store, NULL, NULL)) {
      if(cert_store) X509_STORE_free(cert_store);
      return Status(-1, "Failed to initiate X509_STORE_CTX");
    }

    if (X509_STORE_CTX_get1_issuer(issuer, &store_ctx, cert) <= 0) *issuer = NULL;
    X509_STORE_CTX_cleanup(&store_ctx);
    if(cert_store) X509_STORE_free(cert_store);

    if (*issuer == NULL) return Status(-1, "Can not retrieve issuer certificate");
    *cert_id = OCSP_cert_to_id(NULL, cert, *issuer);

    return Status(0);
  }

//X509_get1_ocsp does not exist before 90807f(0.9.8g)
#if (OPENSSL_VERSION_NUMBER <= 0x0090807fL)
  static int sk_strcmp(const char * const *a, const char * const *b) {
    return strcmp(*a, *b);
  }

  static int append_ia5(STACK **sk, ASN1_IA5STRING *email) {
    char *emtmp;
    /* First some sanity checks */
    if(email->type != V_ASN1_IA5STRING) return 1;
    if(!email->data || !email->length) return 1;
    if(!*sk) *sk = sk_new(sk_strcmp);
    if(!*sk) return 0;
    /* Don't add duplicates */
    if(sk_find(*sk, (char *)email->data) != -1) return 1;
    emtmp = BUF_strdup((char *)email->data);
    if(!emtmp || !sk_push(*sk, emtmp)) {
      X509_email_free(*sk);
      *sk = NULL;
      return 0;
    }
    return 1;
  }

  static STACK* X509_get1_ocsp(X509 *x) {
    AUTHORITY_INFO_ACCESS* info;
    STACK* ret = NULL;
    int i;

    info = (AUTHORITY_INFO_ACCESS*)X509_get_ext_d2i(x, NID_info_access, NULL, NULL);
    if (!info)
      return NULL;
    for (i = 0; i < sk_ACCESS_DESCRIPTION_num(info); i++) {
      ACCESS_DESCRIPTION *ad = sk_ACCESS_DESCRIPTION_value(info, i);
      if (OBJ_obj2nid(ad->method) == NID_ad_OCSP) {
        if (ad->location->type == GEN_URI) {
          if (!append_ia5(&ret, ad->location->d.uniformResourceIdentifier))
            break;
        }
      }
    }
    AUTHORITY_INFO_ACCESS_free(info);
    return ret;
  }
#endif

  static Status get_responder_url(X509* cert, std::string& responder_url) {
#if OPENSSL_VERSION_NUMBER >= 0x10000000L
    STACK_OF(OPENSSL_STRING)* aia = NULL;

    // get AIA information from X509
    aia = X509_get1_ocsp(cert);
    if (aia) //responder_url = sk_OPENSSL_STRING_value(aia, 0);
      responder_url = (OPENSSL_STRING)sk_value(CHECKED_STACK_OF(OPENSSL_STRING, aia), 0);
    else return Status(-1, "No AIA information found");
#else
    STACK_OF(char*)* aia = NULL;

    // get AIA information from X509
    aia = X509_get1_ocsp(cert);
    if (aia) //responder_url = sk_value((const STACK*)CHECKED_PTR_OF(STACK_OF(char*), aia), 0);
      responder_url = sk_value(aia, 0);
    else return Status(-1, "No AIA information found");
#endif

    if (aia) X509_email_free(aia);
    return Status(0);
  }

  typedef struct {
    std::string responder_url;
    Context* context;  // should not be deleted when CerStatusContext is deleted
    AuthN::Utils::Cache* cache;  
    //Cache object is created by ssl_stapling_init_CertStatusContext,
    //therefore it should be deleted when CertStatusContext is deleted
  } CertStatusContext;

  // Use the certificate extension for carrying
  // OCSP stapling related objects
  static int stapling_ex_idx = -1;

  static void ocsp_stapling_obj_free(void* parent, void* ptr, 
        CRYPTO_EX_DATA* ad, int idx, long argl, void* argp) {
    CertStatusContext* cert_stat_ctx = NULL;
    cert_stat_ctx = (CertStatusContext*)ptr;
    if(!cert_stat_ctx) return;
    if(cert_stat_ctx->cache) delete cert_stat_ctx->cache; 
    delete cert_stat_ctx;
    //OPENSSL_free(cert_stat_ctx);
  }

  void ssl_stapling_init(void) {
    if(stapling_ex_idx != -1) return;
    stapling_ex_idx = X509_get_ex_new_index(0, (void*)"X509 cached OCSP info", 0, 0, ocsp_stapling_obj_free);
  }

  Status ssl_stapling_init_CertStatusContext(Context* logctx, SSL_CTX* sslctx, X509* cert, 
      const std::string& cache_file, long cache_interval, const std::string& resp_url) {
    CertStatusContext* cert_stat_ctx = NULL;
    OCSP_CERTID* cert_id = NULL;
    std::string responder_url;
    AuthN::Utils::Cache* cache = NULL;
    Status stat;

    if (cert == NULL) return false;
    cert_stat_ctx  = (CertStatusContext*)X509_get_ex_data(cert, stapling_ex_idx);
    if (cert_stat_ctx) {
      logctx->Log(Context::LogError, "CertStatusContext has already been initialized");
      return Status(-1, "CertStatusContext has already been initialized");
    }
    //cert_stat_ctx = (CertStatusContext*)OPENSSL_malloc(sizeof(CertStatusContext));
    cert_stat_ctx = new CertStatusContext;
    if(!cert_stat_ctx) { 
      logctx->Log(Context::LogError, "Failed to allocate memory for CertStatusContext"); 
      return Status(-1, "Failed to allocate memory for CertStatusContext"); 
    }

    cache = new AuthN::Utils::Cache(cache_file, logctx);
    if(!cache) return Status(-1, "Failed to create Cache");
    if(cache_interval > 0) cache->SetCacheCheckInterval(cache_interval);
   
    // If the url is not given as input, try to find AIA information
    if(resp_url.empty()) {
      stat = get_responder_url(cert, responder_url);
      if((bool)stat) cert_stat_ctx->responder_url = responder_url;
    }
    else cert_stat_ctx->responder_url = resp_url;

    if(cert_stat_ctx->responder_url.empty()) {
      delete cache; return Status(-1, "No AIA and no default responder URL");
    }

    cert_stat_ctx->context = logctx;
    cert_stat_ctx->cache = cache;
    X509_set_ex_data(cert, stapling_ex_idx, cert_stat_ctx);

    return Status(0);
  }

  static Status ssl_stapling_get_CertStatusContext(SSL* ssl, CertStatusContext** cert_stat_ctx) {
    *cert_stat_ctx = NULL;
    X509* cert = NULL;
    cert = SSL_get_certificate(ssl);
    if (cert == NULL) return Status(-1, "X509 object is NULL");
    *cert_stat_ctx = (CertStatusContext*)X509_get_ex_data(cert, stapling_ex_idx);
    if (*cert_stat_ctx && (*cert_stat_ctx)->cache) return Status(0);
    return Status(-1, "OCSP stapling is not suppoted");
  }

  static OCSP_RESPONSE* contact_ocsp_responder(Context& context, OCSP_REQUEST* req, const std::string& url) {
    BIO* cbio = NULL;
    SSL_CTX* ctx = NULL;
    OCSP_RESPONSE* resp = NULL;
    char *host = NULL, *path = NULL, *port = NULL;
    int use_ssl;
    if(!OCSP_parse_url((char*)(url.c_str()), &host, &port, &path, &use_ssl)) {
      context.Log(Context::LogError, "Failed to parse url of OCSP responder");  goto err;
    }
    if(!path) {
      path = (char*)OPENSSL_malloc(2);
      if(!path) {
        context.Log(Context::LogError, "Failed to allocate memory");  goto err;
      }
      path[0] = '/'; path[1] = '\0';
    }
    cbio = BIO_new_connect(host);
    if(!cbio) {
      context.Log(Context::LogError, "Failed to create connect BIO for sending OCSP request");
      goto err;
    }
    if(port) BIO_set_conn_port(cbio, port);
    if(use_ssl) {
      BIO* sbio;
      ctx = SSL_CTX_new(SSLv23_client_method());
      if(ctx == NULL) {
        context.Log(Context::LogError, "Failed to create SSL context"); goto err;
      }
      SSL_CTX_set_mode(ctx, SSL_MODE_AUTO_RETRY);
      sbio = BIO_new_ssl(ctx, 1);
      cbio = BIO_push(sbio, cbio);
    }
    if(BIO_do_connect(cbio) <=0) {
      context.Log(Context::LogError, "Failed when connecting BIO"); goto err;
    }
    resp = OCSP_sendreq_bio(cbio, path, req);
    if(!resp) context.Log(Context::LogError, "Failed to query OCSP responder");

    err:
    if(cbio) BIO_free_all(cbio);
    if(ctx) SSL_CTX_free(ctx);
    if(host) OPENSSL_free(host);
    if(port) OPENSSL_free(port);
    if(path) OPENSSL_free(path);
    if(ctx) SSL_CTX_free(ctx);

    return resp;
  }

/*
#define my_CHECKED_I2D_OF(type, i2d) \
    ((i2d_of_void*) (1 ? i2d : ((I2D_OF(type))0)))

#define my_CHECKED_PTR_OF(type, p) \
    ((void*) (1 ? p : (type*)0))

#define my_ASN1_i2d_bio_of(type,i2d,out,x) \
    (ASN1_i2d_bio(my_CHECKED_I2D_OF(type, (int (*)(type*, unsigned char**))i2d), \
                  out, \
                  (unsigned char*)my_CHECKED_PTR_OF(type, x)))

#define my_i2d_OCSP_RESPONSE_bio(bp,o) my_ASN1_i2d_bio_of(OCSP_RESPONSE,i2d_OCSP_RESPONSE,bp,o)

#define my_i2d_OCSP_REQUEST_bio(bp,o) my_ASN1_i2d_bio_of(OCSP_REQUEST,i2d_OCSP_REQUEST,bp,o)
*/

#if OPENSSL_VERSION_NUMBER >= 0x0090807fL
  #define my_i2d_OCSP_REQUEST_bio(bp,o) \
    ASN1_i2d_bio((i2d_of_void*)i2d_OCSP_REQUEST, bp, (unsigned char*)o)
#elif OPENSSL_VERSION_NUMBER >= 0x00908000L
  #define my_i2d_OCSP_REQUEST_bio(bp,o) \
    ASN1_i2d_bio_of(OCSP_REQUEST,i2d_OCSP_REQUEST, bp, o)
#else
  #define my_i2d_OCSP_REQUEST_bio(bp,o) \
    ASN1_i2d_bio((int (*)())i2d_OCSP_REQUEST, bp, (unsigned char*)o)
#endif

#if OPENSSL_VERSION_NUMBER >= 0x0090807fL
  #define my_i2d_OCSP_RESPONSE_bio(bp,o) \
    ASN1_i2d_bio((i2d_of_void*)i2d_OCSP_RESPONSE, bp, (unsigned char*)o)
#elif OPENSSL_VERSION_NUMBER >= 0x00908000L
  #define my_i2d_OCSP_RESPONSE_bio(bp,o) \
    ASN1_i2d_bio_of(OCSP_RESPONSE,i2d_OCSP_RESPONSE, bp, o)
#else
  #define my_i2d_OCSP_RESPONSE_bio(bp,o) \
    ASN1_i2d_bio((int (*)())i2d_OCSP_RESPONSE, bp, (unsigned char*)o)
#endif


  static bool print_ocsp(BIO *out, OCSP_BASICRESP *bs, OCSP_REQUEST *req,
        OCSP_CERTID *id, long nsec, long maxage) {
    int status, reason;

    ASN1_GENERALIZEDTIME *rev = NULL, *thisupd = NULL, *nextupd = NULL;

    if(!bs || !req || !id) return 1;

    if(!OCSP_resp_find_status(bs, id, &status, &reason, &rev, &thisupd, &nextupd)) {
      BIO_puts(out, "ERROR: No Status found.\n");
      return false;
    }

    if(!OCSP_check_validity(thisupd, nextupd, nsec, maxage)) {
      BIO_puts(out, "WARNING: Status times invalid.\n");
      ERR_print_errors(out);
    }
    BIO_printf(out, "%s\n", OCSP_cert_status_str(status));

    if(thisupd) {
      BIO_puts(out, "\tThis Update: ");
      ASN1_GENERALIZEDTIME_print(out, thisupd);
      BIO_puts(out, "\n");
    }

    if(nextupd) {
      BIO_puts(out, "\tNext Update: ");
      ASN1_GENERALIZEDTIME_print(out, nextupd);
      BIO_puts(out, "\n");
    }

    if(status == V_OCSP_CERTSTATUS_REVOKED) {
      if(reason != -1) BIO_printf(out, "\tReason: %s\n", OCSP_crl_reason_str(reason));
      BIO_puts(out, "\tRevocation Time: ");
      ASN1_GENERALIZEDTIME_print(out, rev);
      BIO_puts(out, "\n");
      return false;
    }
    return true;
  }

  static void bio2string(BIO* in, std::string& out) {
    out.resize(0);
    for(;;) {
      char s[256];
      int l = BIO_read(in,s,sizeof(s));
      if(l <= 0) break;
      out.append(s,l);
    }
  }
 
  static Status ocspreq2string(OCSP_REQUEST* req, std::string& reqstr) {
    unsigned char* req_buf = NULL;
    unsigned char* str;
    unsigned int str_len;
    Status stat;

    if(!req) return Status(-1, "OCSP request is empty");
 
    reqstr.clear();

/*
    str_len = i2d_OCSP_REQUEST(req, NULL);
    if(str_len <= 0) {
      return Status(-1, std::string("OCSP request encoding error") + GetOpenSSLError());
    }
    req_buf = (unsigned char*)OPENSSL_malloc(str_len);
    memset(req_buf, 0, str_len);
    str = req_buf;
    i2d_OCSP_REQUEST(req, &str);
    reqstr.assign((const char*)str, str_len);
*/

    // It seems i2d_OCSP_REQUEST does not process as expected,
    // i2d_OCSP_REQUEST_bio is used instead
    AutoBIO derbio(BIO_new(BIO_s_mem()));
    my_i2d_OCSP_REQUEST_bio(derbio, req);
    bio2string(derbio, reqstr);

    if(req_buf != NULL) OPENSSL_free(req_buf); 
    return Status(0);
  }

  static Status string2ocspreq(const std::string& reqstr, OCSP_REQUEST** req) {
    const char* req_str;
    unsigned int req_str_len;

    req_str = reqstr.c_str();
    req_str_len = reqstr.length();
#if OPENSSL_VERSION_NUMBER >= 0x00908000L
    *req = d2i_OCSP_REQUEST(NULL, (const unsigned char**)&req_str, req_str_len);
#else
    *req = d2i_OCSP_REQUEST(NULL, (unsigned char**)&req_str, req_str_len);
#endif
    if (!*req) {
      return Status(-1, std::string("Unable to parse OCSP request: ") + GetOpenSSLError());
    }
    return Status(0);
  }

  static Status ocspresp2string(OCSP_RESPONSE* resp, std::string& respstr) {
    unsigned char* resp_buf = NULL;
    unsigned char* str;
    unsigned int str_len;
    Status stat;
    
    if(!resp) return Status(-1, "OCSP response is empty");

    respstr.clear();

/*
    str_len = i2d_OCSP_RESPONSE(resp, NULL);
    if(str_len <= 0) {
      return Status(-1, std::string("OCSP response encoding error") + GetOpenSSLError());
    }
    resp_buf = (unsigned char*)OPENSSL_malloc(str_len);
    memset(resp_buf, 0, str_len);
    str = resp_buf;
    i2d_OCSP_RESPONSE(resp, &str);
    respstr.assign((const char*)str, str_len);
*/

    // It seems i2d_OCSP_RESPONSE does not process as expected,
    // i2d_OCSP_RESPONSE_bio is used instead
    AutoBIO derbio(BIO_new(BIO_s_mem()));
    my_i2d_OCSP_RESPONSE_bio(derbio, resp);
    bio2string(derbio, respstr);

    if(resp_buf != NULL) OPENSSL_free(resp_buf);
    return Status(0);
  }

  static Status string2ocspresp(const std::string& respstr, OCSP_RESPONSE** resp) {
    const char* resp_str;
    unsigned int resp_str_len;

    resp_str = respstr.c_str();
    resp_str_len = respstr.length();
#if OPENSSL_VERSION_NUMBER >= 0x00908000L
    *resp = d2i_OCSP_RESPONSE(NULL, (const unsigned char**)&resp_str, resp_str_len);
#else
    *resp = d2i_OCSP_RESPONSE(NULL, (unsigned char**)&resp_str, resp_str_len);
#endif
    if (!*resp) {
      return Status(-1, std::string("Unable to parse OCSP response") + GetOpenSSLError());
    }
    return Status(0);
  }

/*
  static Status string2ocspresp(const std::vector<unsigned char>& respstr, OCSP_RESPONSE** resp) {
    if (respstr.empty()) {
      return Status(-1, "OCSP string is empty");
    }

    unsigned char* resp_str = const_cast<unsigned char*>(&respstr.front());
    unsigned int resp_str_len = respstr.size();

    *resp = d2i_OCSP_RESPONSE(NULL, (const unsigned char**)&resp_str, resp_str_len);
    if (!*resp) {
      return Status(-1, std::string("Unable to parse OCSP response") + GetOpenSSLError());
    }
    return Status(0);
  }
*/

  OCSPContext::~OCSPContext() {
    if(cert_id_) OCSP_CERTID_free(cert_id_); 
    if(issuer_) X509_free(issuer_);
    if(request_signer_) delete request_signer_;
    if(req_) OCSP_REQUEST_free(req_);
    if(resp_) OCSP_RESPONSE_free(resp_);
  }

  OCSPContext::OCSPContext(Context* context, SSL* ssl, AuthN::Utils::Cache* cache, 
        const std::string& resp_url) : context_(context), cert_(NULL), 
        cert_id_(NULL), issuer_(NULL), request_signer_(NULL), ssl_(ssl), 
        cache_(cache), req_(NULL), resp_(NULL) {
    Status stat;
    std::string responder_url;

    cert_ = SSL_get_peer_certificate(ssl_);

    // If the url is not given as input, try to find AIA information
    if(resp_url.empty()) {
      stat = get_responder_url(cert_, responder_url);
      if((bool)stat) ocsp_responder_url_ = responder_url;
    }
    else ocsp_responder_url_ = resp_url;

    stat = get_cert_id(context_, cert_, &issuer_, &cert_id_);

    unsigned char id_str[20];
    unsigned int n;
    X509_digest(cert_, EVP_sha1(), id_str, &n);
    cached_id_.assign((const char*)id_str, n);
  }

  OCSPContext::OCSPContext(Context* ctx, X509* cert, AuthN::Utils::Cache* cache, 
        const std::string& resp_url) : context_(ctx), cert_(cert),
        cert_id_(NULL), issuer_(NULL), request_signer_(NULL), ssl_(NULL),
        cache_(cache), req_(NULL), resp_(NULL) {
    Status stat;
    std::string responder_url;

    // If the url is not given as input, try to find AIA information
    if(resp_url.empty()) {
      stat = get_responder_url(cert_, responder_url);
      if((bool)stat) ocsp_responder_url_ = responder_url;
    }
    else ocsp_responder_url_ = resp_url;

    stat = get_cert_id(context_, cert_, &issuer_, &cert_id_);

    unsigned char id_str[20];
    unsigned int n;
    X509_digest(cert_, EVP_sha1(), id_str, &n);
    cached_id_.assign((const char*)id_str, n);
  }

/*
#if OPENSSL_VERSION_NUMBER <= 0x009080bfL //openssl 0.9.8k
// solve the build issue, because in some relatively older version of openssl:
// ASN1_dup is defined:
// void *ASN1_dup(i2d_of_void *i2d, d2i_of_void *d2i, char *x);
// instead of:
// void *ASN1_dup(i2d_of_void *i2d, d2i_of_void *d2i, void *x);
// but CHECKED_PTR_OF is always defined as:
// #define CHECKED_PTR_OF(type, p) \
    ((void*) (1 ? p : (type*)0))
// so "invalid conversion from 'void*' to 'char*'" occurs

#define my_OCSP_CERTID_dup(cid) \
    (OCSP_CERTID*)ASN1_dup( \
    (i2d_of_void*)i2d_OCSP_CERTID, \
    (d2i_of_void*)d2i_OCSP_CERTID, \
    (char*)cid)
#else
  #define my_OCSP_CERTID_dup(cid) ASN1_dup_of(OCSP_CERTID,i2d_OCSP_CERTID,d2i_OCSP_CERTID,cid)
#endif
*/

#if OPENSSL_VERSION_NUMBER > 0x009080bfL
  #define my_OCSP_CERTID_dup(cid) ASN1_dup_of(OCSP_CERTID,i2d_OCSP_CERTID,d2i_OCSP_CERTID,cid)
#elif OPENSSL_VERSION_NUMBER >= 0x00908000L
  #define my_OCSP_CERTID_dup(cid) \
    (OCSP_CERTID*)ASN1_dup( \
    (i2d_of_void*)i2d_OCSP_CERTID, \
    (d2i_of_void*)d2i_OCSP_CERTID, \
    (char*)cid)
#else
  #define my_OCSP_CERTID_dup(cid) \
    (OCSP_CERTID*)ASN1_dup( \
    (int (*)())i2d_OCSP_CERTID, \
    (char*(*)())d2i_OCSP_CERTID, \
    (char*)cid)
#endif

  Status OCSPContext::MakeOCSPRequest() {
    OCSP_CERTID* id = NULL;
    const EVP_MD* cert_id_md;
    STACK_OF(X509_EXTENSION) *exts;
    X509* signer_cert = NULL;
    EVP_PKEY* signer_key = NULL;
    Status stat;
    int i;

    if(req_ != NULL) OCSP_REQUEST_free(req_);
    req_ = OCSP_REQUEST_new();
    if(!req_) return Status(-1, "Failed to create OCSP request object");
    if(cert_id_ != NULL) {
      id = my_OCSP_CERTID_dup(cert_id_);
      if(!id) { stat = Status(-1, "Failed to duplicate OCSP cert id"); goto err; }
      if (!OCSP_request_add0_id(req_, id)) { 
        stat = Status(-1, "Failed to add OCSP cert id into OCSP request"); goto err; 
      }
      id = NULL;

//#if (OPENSSL_VERSION_NUMBER > 0x009080ffL)
#ifdef HAVE_OCSP_STAPLING
      if(ssl_ != NULL) {
        // Add any extensions to the request
        SSL_get_tlsext_status_exts(ssl_, &exts);
        for (i = 0; i < sk_X509_EXTENSION_num(exts); i++) {
          X509_EXTENSION *ext = sk_X509_EXTENSION_value(exts, i);
          if (!OCSP_REQUEST_add_ext(req_, ext, -1)) goto err;
        }
      }
#endif
    }
    else {  
      stat = get_cert_id(context_, cert_, &issuer_, &cert_id_);
      if(!stat) goto err;
      id = my_OCSP_CERTID_dup(cert_id_);
      if (!OCSP_request_add0_id(req_, id)) { 
        stat = Status(-1, "Failed to add OCSP cert id into OCSP request"); goto err; 
      }
      id = NULL;
    }

    // Sign the request if the signer credential exists
    cert_id_md = EVP_sha1();
    if(request_signer_ != NULL) {
      signer_cert = request_signer_->getCert();
      signer_key = request_signer_->getKey();
      if(signer_cert && signer_key) {
        if(OCSP_request_add1_nonce(req_, NULL, -1) != 1) goto err;
        if(!OCSP_request_sign(req_, signer_cert, signer_key, cert_id_md, NULL, 0)) {
          std::string err = GetOpenSSLError();
          stat = Status(-1, std::string("Failed to sign OCSP request: ") + err);
          context_->LogFormat(Context::LogError, "Failed to sign OCSP request: %s", err.c_str());
          goto err;
        }
      }
    }
   
    stat = Status(0);

    err:
    if(id) OCSP_CERTID_free(id);
    if((stat != Status(0)) && (req_)) { OCSP_REQUEST_free(req_); req_ = NULL; }
    if(stat != Status(0)) context_->Log(Context::LogError, "Failed to create OCSP request");
    last_error_ = stat;
    return stat;
  }
      
  Status OCSPContext::GetOCSPRequest(std::string& reqstr) {
    Status stat;
    stat = ocspreq2string(req_, reqstr);
    return stat;
  }

  Status OCSPContext::RetrieveFromCache() {
    if(cache_ == NULL || !(*cache_)) return Status(-1, "Cache object is not valid");

    std::string retrieved_data;
    Status stat;

    retrieved_data = cache_->RetrieveItem(cached_id_);
    if(retrieved_data.empty()) {
      // Can not retrieve data from cache, is not an error
      context_->Log(Context::LogInfo, "OCSP response cache miss");
      return Status(-1, "OCSP response cache miss");
    }

    if(resp_) { OCSP_RESPONSE_free(resp_); resp_ = NULL; }
    stat = string2ocspresp(retrieved_data, &resp_);
    if(!stat) return stat;

    context_->Log(Context::LogInfo, "OCSP response cache hit");
    return Status(0);
  }

  Status OCSPContext::RetrieveFromCache(std::string& respstr) {
    Status stat;

    // We need the conversion from string to OCSP response object (
    // inside: RetrieveFromCache()) and then
    // the conversion from OCSP response object to string, so that
    // to ensure the cached item is a real OCSP respose string.
    stat = RetrieveFromCache();
    if(stat != Status(0)) return stat;
    stat = ocspresp2string(resp_, respstr);
    
    return stat;
  }

  Status OCSPContext::GetOCSPResponse(std::string& respstr) {
    Status stat;
    stat = ocspresp2string(resp_, respstr);
    return stat;
  }

  Status OCSPContext::StoreToCache() {
    if(cache_ == NULL || !(*cache_)) return Status(-1, "Cache object is not valid");

    std::string stored_data;
    Status stat;

    stat = ocspresp2string(resp_, stored_data);
    if(!stat) return stat;

    long till_time = valid_till_.GetTime();
    cache_->StoreItem(cached_id_, stored_data, till_time);
    context_->Log(Context::LogInfo, "Succeeded to store ocsp response into cache");
    
    return Status(0);
  }

  Status OCSPContext::StoreToCache(const std::string& respstr) {
    if(cache_ == NULL || !(*cache_)) return Status(-1, "Cache object is not valid");
    Status stat;

    if(resp_) { OCSP_RESPONSE_free(resp_); resp_ = NULL; }
    stat = string2ocspresp(respstr, &resp_);
    if(!stat) return stat;

    stat = StoreToCache();

    return stat;
  }

  Status OCSPContext::CheckOCSPResponse() {
    int status, reason;
    long skew = 300; //maximum validity discrepancy in seconds
    long maxage = -1;
    std::string cadir;
    X509_STORE* verify_store = NULL;
    OCSP_BASICRESP* bs = NULL;
    ASN1_GENERALIZEDTIME *rev, *thisupd, *nextupd;
    int res;
    int cert_status;
    Status stat;
    int response_status = OCSP_response_status(resp_);

    cert_status = V_OCSP_CERTSTATUS_UNKNOWN;

    if(response_status != OCSP_RESPONSE_STATUS_SUCCESSFUL) {
      return Status(-1, std::string("OCSP response gives error: ") + OCSP_response_status_str(response_status));
    }

    //verify the OCSP response
    bs = OCSP_response_get1_basic(resp_);
    if (bs == NULL) return Status(-1, "Failed to parse OCSP response");
    if(req_ && ((res = OCSP_check_nonce(req_, bs)) <= 0)) {
      if(res == -1) context_->Log(Context::LogWarning, "There is no nonce in response");
      else {
        stat = Status(-1, "Failed to verify nonce");
        context_->Log(Context::LogError, "Failed to verify nonce"); goto err;
      }
    }

    cadir = context_->GetCAPath();
    if(!cadir.empty()) {
      verify_store = setup_verify(*context_, "", cadir); 
      if(!verify_store) goto err;
      res = OCSP_basic_verify(bs, NULL, verify_store, 0);
      if(res <= 0) {
        context_->LogFormat(Context::LogError, "Failed to verify the response: %s", GetOpenSSLError().c_str());
        stat = Status(-1, std::string("Failed to verify the response: ") + GetOpenSSLError());
        goto err;
      }
      else context_->Log(Context::LogInfo, "Succeeded to verify the response");
    }

    if(!OCSP_resp_find_status(bs, cert_id_, &status, &reason, &rev, &thisupd, &nextupd)) {
      context_->Log(Context::LogWarning, "Failed to retrieve OCSP response status");
      stat = Status(-1, "Failed to retrieve OCSP response status"); goto err;
    }

    //if(status != V_OCSP_CERTSTATUS_UNKNOWN) {
      if (!OCSP_check_validity(thisupd, nextupd, skew, maxage)) {
        stat = Status(-1, "OCSP response is out of validity period");
        goto err;
      }
      else valid_till_ = asn1_to_utctime(nextupd);
    //}

    context_->LogFormat(Context::LogInfo, "OCSP validation completed. Certificate status: %s, reason: %s", OCSP_cert_status_str(status), OCSP_crl_reason_str(reason));
    cert_status = status;

    stat = Status(0);

    err:
    OCSP_BASICRESP_free(bs);
    if(cert_status == V_OCSP_CERTSTATUS_GOOD) cert_status_ = OCSPContext::CertStatusGood;
    else if(cert_status == V_OCSP_CERTSTATUS_REVOKED) cert_status_ = OCSPContext::CertStatusRevoked;
    else cert_status_ = OCSPContext::CertStatusUnknown;
    return stat;
  }
 
  Status OCSPContext::CheckOCSPResponse(const std::string& reqstr, const std::string& respstr) {
    if(respstr.empty()) return Status(-1, "OCSP response is empty");

    Status stat;

    if(!reqstr.empty()) {
      if(req_) OCSP_REQUEST_free(req_); req_ = NULL;
      stat = string2ocspreq(reqstr, &req_);
      if(!stat) return stat;
    }

    if(resp_) { OCSP_RESPONSE_free(resp_); resp_ = NULL; }
    stat = string2ocspresp(respstr, &resp_);
    if(!stat) return stat;

    stat = CheckOCSPResponse();

    return stat;
  }

  Status OCSPContext::CheckOCSPResponse(const std::string& respstr) {
    Status stat;
    stat = CheckOCSPResponse("", respstr);
    return stat;
  }
 
  Status OCSPContext::QueryOCSPResponder(const std::string& responder_url) {
    std::string ocsp_resp_out;
    AutoBIO ocsp_resp_out_bio(BIO_new(BIO_s_mem()));
    OCSP_CERTID* id = NULL;
    STACK_OF(X509_EXTENSION) *exts;
    Status stat;
    int i;

    if(!req_) return Status(-1, "OCSP request is empty, call MakeOCSPRequest at first");

    if(resp_) { OCSP_RESPONSE_free(resp_); resp_ = NULL; }

    if(!responder_url.empty())
      resp_ = contact_ocsp_responder(*context_, req_, responder_url);
    else if(!ocsp_responder_url_.empty())
      resp_ = contact_ocsp_responder(*context_, req_, ocsp_responder_url_);
    if (!resp_) {
      context_->Log(Context::LogError, "Failed to query OCSP responder");
      last_error_ = Status(-1, "Failed to query OCSP responder");
      goto done;
    }

    // Check the validity of the OCSP response
    stat = CheckOCSPResponse();
    if(!stat) { 
      context_->Log(Context::LogError, "OCSP response is not valid");
      last_error_ = Status(-1, "OCSP response is not valid"); goto done; 
    }

    // Store the OCSP response into cache
    // Use the "next Update" as the expiration time
    // of the cache item
    stat = StoreToCache();
    if(!stat) {
      last_error_ = Status(-1, "Failed to store OCSP response into cache"); goto done; 
    }

    OCSP_RESPONSE_print(ocsp_resp_out_bio, resp_, 2);
    bio2string(ocsp_resp_out_bio, ocsp_resp_out);
    context_->LogFormat(Context::LogVerbose, "OCSP response received: \n %s", ocsp_resp_out.c_str());

    last_error_ = Status(0);

    done:
    if (last_error_ != Status(0)) context_->LogFormat(Context::LogError, GetOpenSSLError());
    if (id) OCSP_CERTID_free(id);
    return last_error_;

    err:
    last_error_ = Status(-2, "Fatal error happens"); 
    goto done;
  }

/*
  Status OCSPContext::QueryOCSPResponder(const std::string& reqstr, std::string& respstr, const std::string& responder_url) {
    Status stat;
    
    if(!reqstr.empty()) {
      if(req_) OCSP_REQUEST_free(req_);
      stat = string2ocspreq(reqstr, &req_);
      if(!stat) return stat;
    }

    stat = QueryOCSPResponder(responder_url);
    if(!stat) {
      if(req_) OCSP_REQUEST_free(req_);
      if(resp_) OCSP_RESPONSE_free(resp_);
      return stat;
    }

    stat = ocspresp2string(resp_, respstr);

    return stat;
  }
*/

  void OCSPContext::SetRequestSigner(const std::string& certstr, const std::string& keystr) {
    request_signer_ = new CertContext(*context_, certstr, keystr);
    if(!request_signer_ || !(*request_signer_)) return;
  }

  void OCSPContext::SetRequestSigner(const char* certfile, const char* keyfile) {
    request_signer_ = new CertContext(*context_, certfile, keyfile);
    if(!request_signer_ || !(*request_signer_)) return;
  }

//#if (OPENSSL_VERSION_NUMBER > 0x009080ffL)
#ifdef HAVE_OCSP_STAPLING
  // server callback and client callback are only
  // valid for the openssl versions on which tls ext data
  // is supported. 
  int ocsp_stapling_server_callback(SSL* ssl, void* arg) {
    int ret = SSL_TLSEXT_ERR_NOACK;
    CertStatusContext* csctx = NULL;
    Context* context = NULL;
    AuthN::Utils::Cache* cache = NULL;
    OCSP_CERTID* cert_id = NULL;
    X509* cert = NULL;
    std::string responder_url;
    std::string req_str, resp_str;
    OCSPContext::CertStatus cert_status;
    Status stat;

    stat =  ssl_stapling_get_CertStatusContext(ssl, &csctx);
    if(!stat) return SSL_TLSEXT_ERR_NOACK;

    context = csctx->context;
    cache  = csctx->cache;
    responder_url = csctx->responder_url;

    context->Log(Context::LogVerbose, "OCSP stapling server: callback called");
 
    cert = SSL_get_certificate(ssl); 

    OCSPContext ocsp_ctx(context, cert, cache, responder_url);
    if(context != NULL) {
      std::string cert_path = context->GetCertPath();
      std::string key_path = context->GetKeyPath();
      ocsp_ctx.SetRequestSigner(cert_path.c_str(), key_path.c_str());
    }

    stat = ocsp_ctx.RetrieveFromCache();
    if(stat) {
      stat = ocsp_ctx.CheckOCSPResponse();
      if(!stat) {
        // Need to contact responder
        //context->Log(Context::LogError, "OCSP response in cache is not valid");
        context->Log(Context::LogError, stat.GetDescription());
      }
      else stat = ocsp_ctx.GetOCSPResponse(resp_str);
    }
    else context->Log(Context::LogError, stat.GetDescription());

    if(!stat) {
      stat = ocsp_ctx.MakeOCSPRequest();
      if(!stat) {
        context->Log(Context::LogError, stat.GetDescription());
        return SSL_TLSEXT_ERR_ALERT_FATAL;
      }
      stat = ocsp_ctx.QueryOCSPResponder(responder_url);
      if(!stat) context->Log(Context::LogError, stat.GetDescription());
      if(stat == Status(-2)) return SSL_TLSEXT_ERR_ALERT_FATAL;
      stat = ocsp_ctx.GetOCSPResponse(resp_str);

      if(!stat) context->Log(Context::LogError, stat.GetDescription());
    }

    if(stat) {
      if(resp_str.empty()) return SSL_TLSEXT_ERR_ALERT_FATAL;

      char* str = NULL;
      int len = resp_str.length();
      str = (char*)OPENSSL_malloc(len);
      memset((void*)str, 0, len);
      memcpy((void*)str, resp_str.c_str(), len);

      //Set the ocsp response into ssl connection, so that the
      //verifier can retrieve it on the peer by using: SSL_get_tlsext_status_ocsp_resp
      SSL_set_tlsext_status_ocsp_resp(ssl, str, len);
     
      // The allocated string should not be deallocated, because
      // SSL_set_tlsext_status_ocsp_resp simply use the point to 
      // this string, rather than copy the content.
      //OPENSSL_free(str);
      return SSL_TLSEXT_ERR_OK;

    }
    context->Log(Context::LogInfo, "No OCSP response available");

    return SSL_TLSEXT_ERR_NOACK;

  }

  // Get the ocsp response with ocsp stapling,
  // and store the valid response into local cache.
  int ocsp_stapling_client_callback(SSL *s, void *arg) {
    Context* context = (Context*)arg;
    const unsigned char *p;
    int len;
    OCSP_RESPONSE *rsp = NULL;
    std::string ocsp_resp;
    AutoBIO bio(BIO_new(BIO_s_mem()));


    context->Log(Context::LogVerbose, "OCSP stapling client: callback called");

    len = SSL_get_tlsext_status_ocsp_resp(s, &p);
    if (!p) {
      context->Log(Context::LogWarning, "No OCSP response received");
      return 1;
    }
 
    // Note d2i_OCSP_RESPONSE seems to change the content of the string,
    // therefore the string should be assigned to some place in
    // advance if needed.
    ocsp_resp.assign((const char*)p, (size_t)len);
    rsp = d2i_OCSP_RESPONSE(NULL, &p, len);
    if (!rsp) {
      context->Log(Context::LogWarning, "Failed to parse OCSP response");
      BIO_dump_indent(bio, (char *)p, len, 4);
      return 0;
    }
    OCSP_RESPONSE_print(bio, rsp, 0);
    OCSP_RESPONSE_free(rsp);
    std::string str;
    bio2string(bio, str);
    context->Log(Context::LogVerbose, "The OCSP response:");
    context->Log(Context::LogVerbose, str);

    //X509_STORE_CTX_set_error(ctx, X509_V_ERR_CERT_REVOKED);
    //SSL_set_verify_result(s,X509_V_ERR_CERT_REVOKED);
    //SSL_set_app_data(s, (char*)1);
    //char* r = (char*)SSL_get_app_data(s);

    // TODO: get cache location from where?
    const std::string ocsp_cache_file = "/tmp/ocsp_cache";
    std::string capath = context->GetCAPath();
    AuthN::Utils::Cache* ocsp_cache = new AuthN::Utils::Cache(ocsp_cache_file, context);
    OCSPContext ocsp_ctx(context, s, ocsp_cache);
    //ocsp_ctx.SetRequestSigner(cert_path.c_str(), key_path.c_str());
    std::string req_str;
    Status status;
    status = ocsp_ctx.CheckOCSPResponse(ocsp_resp);
    if(!status) context->Log(Context::LogWarning, status.GetDescription());
    else {
      status = ocsp_ctx.StoreToCache(ocsp_resp);
      if(!status) context->Log(Context::LogWarning, status.GetDescription());
    }
    delete ocsp_cache;

    if(status) context->Log(Context::LogInfo, "The ocsp response has been verified and cached");
    return 1;
  }

#endif

}
}
